/*
 * Decompiled with CFR 0.152.
 */
package com.jinmingyunle.midiplaylib.midifile;

import com.jinmingyunle.midiplaylib.file.FileUri;
import com.jinmingyunle.midiplaylib.midifile.MidiEvent;
import com.jinmingyunle.midiplaylib.midifile.MidiFileException;
import com.jinmingyunle.midiplaylib.midifile.MidiFileReader;
import com.jinmingyunle.midiplaylib.midifile.MidiNote;
import com.jinmingyunle.midiplaylib.midifile.MidiOptions;
import com.jinmingyunle.midiplaylib.midifile.MidiTrack;
import com.jinmingyunle.midiplaylib.midifile.PairInt;
import com.jinmingyunle.midiplaylib.musictools.TimeSignature;
import com.jinmingyunle.midiplaylib.utils.ListInt;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

public class MidiFile {
    private static final String TAG = "MidiFileTAG";
    private FileUri fileuri;
    private String filename;
    private ArrayList<ArrayList<MidiEvent>> allevents;
    private ArrayList<MidiTrack> tracks;
    private short trackmode;
    private TimeSignature timesig;
    private int quarternote;
    private int totalpulses;
    private boolean trackPerChannel;
    public static final byte EventNoteOff = -128;
    public static final byte EventNoteOn = -112;
    public static final byte EventKeyPressure = -96;
    public static final byte EventControlChange = -80;
    public static final byte EventProgramChange = -64;
    public static final byte EventChannelPressure = -48;
    public static final byte EventPitchBend = -32;
    public static final byte SysexEvent1 = -16;
    public static final byte SysexEvent2 = -9;
    public static final byte MetaEvent = -1;
    public static final byte MetaEventSequence = 0;
    public static final byte MetaEventText = 1;
    public static final byte MetaEventCopyright = 2;
    public static final byte MetaEventSequenceName = 3;
    public static final byte MetaEventInstrument = 4;
    public static final byte MetaEventLyric = 5;
    public static final byte MetaEventMarker = 6;
    public static final byte MetaEventEndOfTrack = 47;
    public static final byte MetaEventTempo = 81;
    public static final byte MetaEventSMPTEOffset = 84;
    public static final byte MetaEventTimeSignature = 88;
    public static final byte MetaEventKeySignature = 89;
    public static String[] Instruments = new String[]{"Acoustic Grand Piano", "Bright Acoustic Piano", "Electric Grand Piano", "Honky-tonk Piano", "Electric Piano 1", "Electric Piano 2", "Harpsichord", "Clavi", "Celesta", "Glockenspiel", "Music Box", "Vibraphone", "Marimba", "Xylophone", "Tubular Bells", "Dulcimer", "Drawbar Organ", "Percussive Organ", "Rock Organ", "Church Organ", "Reed Organ", "Accordion", "Harmonica", "Tango Accordion", "Acoustic Guitar (nylon)", "Acoustic Guitar (steel)", "Electric Guitar (jazz)", "Electric Guitar (clean)", "Electric Guitar (muted)", "Overdriven Guitar", "Distortion Guitar", "Guitar harmonics", "Acoustic Bass", "Electric Bass (finger)", "Electric Bass (pick)", "Fretless Bass", "Slap Bass 1", "Slap Bass 2", "Synth Bass 1", "Synth Bass 2", "Violin", "Viola", "Cello", "Contrabass", "Tremolo Strings", "Pizzicato Strings", "Orchestral Harp", "Timpani", "String Ensemble 1", "String Ensemble 2", "SynthStrings 1", "SynthStrings 2", "Choir Aahs", "Voice Oohs", "Synth Voice", "Orchestra Hit", "Trumpet", "Trombone", "Tuba", "Muted Trumpet", "French Horn", "Brass Section", "SynthBrass 1", "SynthBrass 2", "Soprano Sax", "Alto Sax", "Tenor Sax", "Baritone Sax", "Oboe", "English Horn", "Bassoon", "Clarinet", "Piccolo", "Flute", "Recorder", "Pan Flute", "Blown Bottle", "Shakuhachi", "Whistle", "Ocarina", "Lead 1 (square)", "Lead 2 (sawtooth)", "Lead 3 (calliope)", "Lead 4 (chiff)", "Lead 5 (charang)", "Lead 6 (voice)", "Lead 7 (fifths)", "Lead 8 (bass + lead)", "Pad 1 (new age)", "Pad 2 (warm)", "Pad 3 (polysynth)", "Pad 4 (choir)", "Pad 5 (bowed)", "Pad 6 (metallic)", "Pad 7 (halo)", "Pad 8 (sweep)", "FX 1 (rain)", "FX 2 (soundtrack)", "FX 3 (crystal)", "FX 4 (atmosphere)", "FX 5 (brightness)", "FX 6 (goblins)", "FX 7 (echoes)", "FX 8 (sci-fi)", "Sitar", "Banjo", "Shamisen", "Koto", "Kalimba", "Bag pipe", "Fiddle", "Shanai", "Tinkle Bell", "Agogo", "Steel Drums", "Woodblock", "Taiko Drum", "Melodic Tom", "Synth Drum", "Reverse Cymbal", "Guitar Fret Noise", "Breath Noise", "Seashore", "Bird Tweet", "Telephone Ring", "Helicopter", "Applause", "Gunshot", "Percussion"};

    private String EventName(int ev) {
        if (ev >= -128 && ev < -112) {
            return "NoteOff";
        }
        if (ev >= -112 && ev < -96) {
            return "NoteOn";
        }
        if (ev >= -96 && ev < -80) {
            return "KeyPressure";
        }
        if (ev >= -80 && ev < -64) {
            return "ControlChange";
        }
        if (ev >= -64 && ev < -48) {
            return "ProgramChange";
        }
        if (ev >= -48 && ev < -32) {
            return "ChannelPressure";
        }
        if (ev >= -32 && ev < -16) {
            return "PitchBend";
        }
        if (ev == -1) {
            return "MetaEvent";
        }
        if (ev == -16 || ev == -9) {
            return "SysexEvent";
        }
        return "Unknown";
    }

    private String MetaName(int ev) {
        if (ev == 0) {
            return "MetaEventSequence";
        }
        if (ev == 1) {
            return "MetaEventText";
        }
        if (ev == 2) {
            return "MetaEventCopyright";
        }
        if (ev == 3) {
            return "MetaEventSequenceName";
        }
        if (ev == 4) {
            return "MetaEventInstrument";
        }
        if (ev == 5) {
            return "MetaEventLyric";
        }
        if (ev == 6) {
            return "MetaEventMarker";
        }
        if (ev == 47) {
            return "MetaEventEndOfTrack";
        }
        if (ev == 81) {
            return "MetaEventTempo";
        }
        if (ev == 84) {
            return "MetaEventSMPTEOffset";
        }
        if (ev == 88) {
            return "MetaEventTimeSignature";
        }
        if (ev == 89) {
            return "MetaEventKeySignature";
        }
        return "Unknown";
    }

    public ArrayList<MidiTrack> getTracks() {
        return this.tracks;
    }

    public TimeSignature getTime() {
        return this.timesig;
    }

    public String getFileName() {
        return this.filename;
    }

    public int getTotalPulses() {
        return this.totalpulses;
    }

    public MidiFile(byte[] rawdata, String filename) {
        this.filename = filename;
        this.parse(rawdata);
    }

    private void parse(byte[] rawdata) {
        this.tracks = new ArrayList();
        this.trackPerChannel = false;
        MidiFileReader file = new MidiFileReader(rawdata);
        String id = file.ReadAscii(4);
        if (!id.equals("MThd")) {
            throw new MidiFileException("Doesn't start with MThd", 0);
        }
        int len = file.ReadInt();
        if (len != 6) {
            throw new MidiFileException("Bad MThd header", 4);
        }
        this.trackmode = (short)file.ReadShort();
        int num_tracks = file.ReadShort();
        this.quarternote = file.ReadShort();
        this.allevents = new ArrayList();
        for (int tracknum = 0; tracknum < num_tracks; ++tracknum) {
            this.allevents.add(this.ReadTrack(file));
            MidiTrack track = new MidiTrack(this.allevents.get(tracknum), tracknum);
            if (track.getNotes().size() <= 0) continue;
            this.tracks.add(track);
        }
        for (MidiTrack track : this.tracks) {
            MidiNote last = track.getNotes().get(track.getNotes().size() - 1);
            if (this.totalpulses >= last.getStartTime() + last.getDuration()) continue;
            this.totalpulses = last.getStartTime() + last.getDuration();
        }
        if (this.tracks.size() == 1 && MidiFile.HasMultipleChannels(this.tracks.get(0))) {
            this.tracks = MidiFile.SplitChannels(this.tracks.get(0), this.allevents.get(this.tracks.get(0).trackNumber()));
            this.trackPerChannel = true;
        }
        MidiFile.CheckStartTimes(this.tracks);
        int tempoCount = 0;
        long tempo = 0L;
        int numer = 0;
        int denom = 0;
        for (ArrayList<MidiEvent> list : this.allevents) {
            for (MidiEvent mevent : list) {
                if (mevent.Metaevent == 81) {
                    tempo += (long)mevent.Tempo;
                    ++tempoCount;
                }
                if (mevent.Metaevent != 88 || numer != 0) continue;
                numer = mevent.Numerator;
                denom = mevent.Denominator;
            }
        }
        tempo = tempo == 0L ? 500000L : (tempo /= (long)tempoCount);
        if (numer == 0) {
            numer = 4;
            denom = 4;
        }
        this.timesig = new TimeSignature(numer, denom, this.quarternote, (int)tempo);
    }

    private ArrayList<MidiEvent> ReadTrack(MidiFileReader file) {
        ArrayList<MidiEvent> result = new ArrayList<MidiEvent>(20);
        int starttime = 0;
        String id = file.ReadAscii(4);
        if (!id.equals("MTrk")) {
            throw new MidiFileException("Bad MTrk header", file.GetOffset() - 4);
        }
        int tracklen = file.ReadInt();
        int trackend = tracklen + file.GetOffset();
        int eventflag = 0;
        while (file.GetOffset() < trackend) {
            byte peekevent;
            int deltatime;
            try {
                int startoffset = file.GetOffset();
                deltatime = file.ReadVarlen();
                starttime += deltatime;
                peekevent = file.Peek();
            }
            catch (MidiFileException e) {
                return result;
            }
            MidiEvent mevent = new MidiEvent();
            result.add(mevent);
            mevent.DeltaTime = deltatime;
            mevent.StartTime = starttime;
            if (peekevent < 0) {
                mevent.HasEventflag = true;
                eventflag = file.ReadByte();
            }
            if (eventflag >= -112 && eventflag < -96) {
                mevent.EventFlag = (byte)-112;
                mevent.Channel = (byte)(eventflag - -112);
                mevent.Notenumber = file.ReadByte();
                mevent.Velocity = file.ReadByte();
                continue;
            }
            if (eventflag >= -128 && eventflag < -112) {
                mevent.EventFlag = (byte)-128;
                mevent.Channel = (byte)(eventflag - -128);
                mevent.Notenumber = file.ReadByte();
                mevent.Velocity = file.ReadByte();
                continue;
            }
            if (eventflag >= -96 && eventflag < -80) {
                mevent.EventFlag = (byte)-96;
                mevent.Channel = (byte)(eventflag - -96);
                mevent.Notenumber = file.ReadByte();
                mevent.KeyPressure = file.ReadByte();
                continue;
            }
            if (eventflag >= -80 && eventflag < -64) {
                mevent.EventFlag = (byte)-80;
                mevent.Channel = (byte)(eventflag - -80);
                mevent.ControlNum = file.ReadByte();
                mevent.ControlValue = file.ReadByte();
                continue;
            }
            if (eventflag >= -64 && eventflag < -48) {
                mevent.EventFlag = (byte)-64;
                mevent.Channel = (byte)(eventflag - -64);
                mevent.Instrument = file.ReadByte();
                continue;
            }
            if (eventflag >= -48 && eventflag < -32) {
                mevent.EventFlag = (byte)-48;
                mevent.Channel = (byte)(eventflag - -48);
                mevent.ChanPressure = file.ReadByte();
                continue;
            }
            if (eventflag >= -32 && eventflag < -16) {
                mevent.EventFlag = (byte)-32;
                mevent.Channel = (byte)(eventflag - -32);
                mevent.PitchBend = (short)file.ReadShort();
                continue;
            }
            if (eventflag == -16) {
                mevent.EventFlag = (byte)-16;
                mevent.Metalength = file.ReadVarlen();
                mevent.Value = file.ReadBytes(mevent.Metalength);
                continue;
            }
            if (eventflag == -9) {
                mevent.EventFlag = (byte)-9;
                mevent.Metalength = file.ReadVarlen();
                mevent.Value = file.ReadBytes(mevent.Metalength);
                continue;
            }
            if (eventflag == -1) {
                mevent.EventFlag = (byte)-1;
                mevent.Metaevent = file.ReadByte();
                mevent.Metalength = file.ReadVarlen();
                mevent.Value = file.ReadBytes(mevent.Metalength);
                if (mevent.Metaevent == 88) {
                    if (mevent.Metalength < 2) {
                        throw new MidiFileException("Meta Event Time Signature len == " + mevent.Metalength + " != 4", file.GetOffset());
                    }
                    mevent.Numerator = mevent.Value[0];
                    mevent.Denominator = (byte)Math.pow(2.0, mevent.Value[1]);
                    continue;
                }
                if (mevent.Metaevent == 81) {
                    if (mevent.Metalength != 3) {
                        throw new MidiFileException("Meta Event Tempo len == " + mevent.Metalength + " != 3", file.GetOffset());
                    }
                    mevent.Tempo = (mevent.Value[0] & 0xFF) << 16 | (mevent.Value[1] & 0xFF) << 8 | mevent.Value[2] & 0xFF;
                    continue;
                }
                if (mevent.Metaevent != 47) continue;
                continue;
            }
            throw new MidiFileException("Unknown event " + mevent.EventFlag, file.GetOffset() - 1);
        }
        return result;
    }

    static boolean HasMultipleChannels(MidiTrack track) {
        int channel = track.getNotes().get(0).getChannel();
        for (MidiNote note : track.getNotes()) {
            if (note.getChannel() == channel) continue;
            return true;
        }
        return false;
    }

    static int VarlenToBytes(int num, byte[] buf, int offset) {
        byte b1 = (byte)(num >> 21 & 0x7F);
        byte b2 = (byte)(num >> 14 & 0x7F);
        byte b3 = (byte)(num >> 7 & 0x7F);
        byte b4 = (byte)(num & 0x7F);
        if (b1 > 0) {
            buf[offset] = (byte)(b1 | 0x80);
            buf[offset + 1] = (byte)(b2 | 0x80);
            buf[offset + 2] = (byte)(b3 | 0x80);
            buf[offset + 3] = b4;
            return 4;
        }
        if (b2 > 0) {
            buf[offset] = (byte)(b2 | 0x80);
            buf[offset + 1] = (byte)(b3 | 0x80);
            buf[offset + 2] = b4;
            return 3;
        }
        if (b3 > 0) {
            buf[offset] = (byte)(b3 | 0x80);
            buf[offset + 1] = b4;
            return 2;
        }
        buf[offset] = b4;
        return 1;
    }

    private static void IntToBytes(int value, byte[] data, int offset) {
        data[offset] = (byte)(value >> 24 & 0xFF);
        data[offset + 1] = (byte)(value >> 16 & 0xFF);
        data[offset + 2] = (byte)(value >> 8 & 0xFF);
        data[offset + 3] = (byte)(value & 0xFF);
    }

    private static int GetTrackLength(ArrayList<MidiEvent> events) {
        int len = 0;
        byte[] buf = new byte[1024];
        for (MidiEvent mevent : events) {
            len += MidiFile.VarlenToBytes(mevent.DeltaTime, buf, 0);
            ++len;
            switch (mevent.EventFlag) {
                case -112: {
                    len += 2;
                    break;
                }
                case -128: {
                    len += 2;
                    break;
                }
                case -96: {
                    len += 2;
                    break;
                }
                case -80: {
                    len += 2;
                    break;
                }
                case -64: {
                    ++len;
                    break;
                }
                case -48: {
                    ++len;
                    break;
                }
                case -32: {
                    len += 2;
                    break;
                }
                case -16: 
                case -9: {
                    len += MidiFile.VarlenToBytes(mevent.Metalength, buf, 0);
                    len += mevent.Metalength;
                    break;
                }
                case -1: {
                    ++len;
                    len += MidiFile.VarlenToBytes(mevent.Metalength, buf, 0);
                    len += mevent.Metalength;
                    break;
                }
            }
        }
        return len;
    }

    private static void ArrayCopy(byte[] src, int srcoffset, byte[] dest, int destoffset, int len) {
        for (int i = 0; i < len; ++i) {
            dest[destoffset + i] = src[srcoffset + i];
        }
    }

    private static void WriteEvents(FileOutputStream file, ArrayList<ArrayList<MidiEvent>> allevents, int trackmode, int quarter) throws IOException {
        byte[] buf = new byte[16384];
        file.write("MThd".getBytes("US-ASCII"), 0, 4);
        MidiFile.IntToBytes(6, buf, 0);
        file.write(buf, 0, 4);
        buf[0] = (byte)(trackmode >> 8);
        buf[1] = (byte)(trackmode & 0xFF);
        file.write(buf, 0, 2);
        buf[0] = 0;
        buf[1] = (byte)allevents.size();
        file.write(buf, 0, 2);
        buf[0] = (byte)(quarter >> 8);
        buf[1] = (byte)(quarter & 0xFF);
        file.write(buf, 0, 2);
        for (ArrayList<MidiEvent> list : allevents) {
            file.write("MTrk".getBytes("US-ASCII"), 0, 4);
            int len = MidiFile.GetTrackLength(list);
            MidiFile.IntToBytes(len, buf, 0);
            file.write(buf, 0, 4);
            for (MidiEvent mevent : list) {
                int offset;
                int varlen = MidiFile.VarlenToBytes(mevent.DeltaTime, buf, 0);
                file.write(buf, 0, varlen);
                buf[0] = mevent.EventFlag == -16 || mevent.EventFlag == -9 || mevent.EventFlag == -1 ? mevent.EventFlag : (byte)(mevent.EventFlag + mevent.Channel);
                file.write(buf, 0, 1);
                if (mevent.EventFlag == -112) {
                    buf[0] = mevent.Notenumber;
                    buf[1] = mevent.Velocity;
                    file.write(buf, 0, 2);
                    continue;
                }
                if (mevent.EventFlag == -128) {
                    buf[0] = mevent.Notenumber;
                    buf[1] = mevent.Velocity;
                    file.write(buf, 0, 2);
                    continue;
                }
                if (mevent.EventFlag == -96) {
                    buf[0] = mevent.Notenumber;
                    buf[1] = mevent.KeyPressure;
                    file.write(buf, 0, 2);
                    continue;
                }
                if (mevent.EventFlag == -80) {
                    buf[0] = mevent.ControlNum;
                    buf[1] = mevent.ControlValue;
                    file.write(buf, 0, 2);
                    continue;
                }
                if (mevent.EventFlag == -64) {
                    buf[0] = mevent.Instrument;
                    file.write(buf, 0, 1);
                    continue;
                }
                if (mevent.EventFlag == -48) {
                    buf[0] = mevent.ChanPressure;
                    file.write(buf, 0, 1);
                    continue;
                }
                if (mevent.EventFlag == -32) {
                    buf[0] = (byte)(mevent.PitchBend >> 8);
                    buf[1] = (byte)(mevent.PitchBend & 0xFF);
                    file.write(buf, 0, 2);
                    continue;
                }
                if (mevent.EventFlag == -16) {
                    offset = MidiFile.VarlenToBytes(mevent.Metalength, buf, 0);
                    MidiFile.ArrayCopy(mevent.Value, 0, buf, offset, mevent.Value.length);
                    file.write(buf, 0, offset + mevent.Value.length);
                    continue;
                }
                if (mevent.EventFlag == -9) {
                    offset = MidiFile.VarlenToBytes(mevent.Metalength, buf, 0);
                    MidiFile.ArrayCopy(mevent.Value, 0, buf, offset, mevent.Value.length);
                    file.write(buf, 0, offset + mevent.Value.length);
                    continue;
                }
                if (mevent.EventFlag == -1 && mevent.Metaevent == 81) {
                    buf[0] = mevent.Metaevent;
                    buf[1] = 3;
                    buf[2] = (byte)(mevent.Tempo >> 16 & 0xFF);
                    buf[3] = (byte)(mevent.Tempo >> 8 & 0xFF);
                    buf[4] = (byte)(mevent.Tempo & 0xFF);
                    file.write(buf, 0, 5);
                    continue;
                }
                if (mevent.EventFlag != -1) continue;
                buf[0] = mevent.Metaevent;
                offset = MidiFile.VarlenToBytes(mevent.Metalength, buf, 1) + 1;
                MidiFile.ArrayCopy(mevent.Value, 0, buf, offset, mevent.Value.length);
                file.write(buf, 0, offset + mevent.Value.length);
            }
        }
        file.close();
    }

    private static ArrayList<ArrayList<MidiEvent>> CloneMidiEvents(ArrayList<ArrayList<MidiEvent>> origlist) {
        ArrayList<ArrayList<MidiEvent>> newlist = new ArrayList<ArrayList<MidiEvent>>(origlist.size());
        for (int tracknum = 0; tracknum < origlist.size(); ++tracknum) {
            ArrayList<MidiEvent> origevents = origlist.get(tracknum);
            ArrayList<MidiEvent> newevents = new ArrayList<MidiEvent>(origevents.size());
            newlist.add(newevents);
            for (MidiEvent mevent : origevents) {
                newevents.add(mevent.Clone());
            }
        }
        return newlist;
    }

    private static MidiEvent CreateTempoEvent(int tempo) {
        MidiEvent mevent = new MidiEvent();
        mevent.DeltaTime = 0;
        mevent.StartTime = 0;
        mevent.HasEventflag = true;
        mevent.EventFlag = (byte)-1;
        mevent.Metaevent = (byte)81;
        mevent.Metalength = 3;
        mevent.Tempo = tempo;
        return mevent;
    }

    private static void UpdateControlChange(ArrayList<MidiEvent> newevents, MidiEvent changeEvent) {
        for (MidiEvent mevent : newevents) {
            if (mevent.EventFlag != changeEvent.EventFlag || mevent.Channel != changeEvent.Channel || mevent.ControlNum != changeEvent.ControlNum) continue;
            mevent.ControlValue = changeEvent.ControlValue;
            return;
        }
        newevents.add(changeEvent);
    }

    private static ArrayList<ArrayList<MidiEvent>> StartAtPauseTime(ArrayList<ArrayList<MidiEvent>> list, int pauseTime) {
        ArrayList<ArrayList<MidiEvent>> newlist = new ArrayList<ArrayList<MidiEvent>>(list.size());
        for (int tracknum = 0; tracknum < list.size(); ++tracknum) {
            ArrayList<MidiEvent> events = list.get(tracknum);
            ArrayList<MidiEvent> newevents = new ArrayList<MidiEvent>(events.size());
            newlist.add(newevents);
            boolean foundEventAfterPause = false;
            for (MidiEvent mevent : events) {
                if (mevent.StartTime < pauseTime) {
                    if (mevent.EventFlag == -112 || mevent.EventFlag == -128) continue;
                    if (mevent.EventFlag == -80) {
                        mevent.DeltaTime = 0;
                        MidiFile.UpdateControlChange(newevents, mevent);
                        continue;
                    }
                    mevent.DeltaTime = 0;
                    newevents.add(mevent);
                    continue;
                }
                if (!foundEventAfterPause) {
                    mevent.DeltaTime = mevent.StartTime - pauseTime;
                    newevents.add(mevent);
                    foundEventAfterPause = true;
                    continue;
                }
                newevents.add(mevent);
            }
        }
        return newlist;
    }

    public void ChangeSound(FileOutputStream destfile, MidiOptions options) throws IOException {
        this.Write(destfile, options);
    }

    public void Write(FileOutputStream destfile, MidiOptions options) throws IOException {
        ArrayList<ArrayList<MidiEvent>> newevents = this.allevents;
        if (options != null) {
            newevents = this.ApplyOptionsToEvents(options);
        }
        MidiFile.WriteEvents(destfile, newevents, this.trackmode, this.quarternote);
    }

    public ArrayList<ArrayList<MidiEvent>> ApplyOptionsToEvents(MidiOptions options) {
        int tracknum;
        int i;
        if (this.trackPerChannel) {
            return this.ApplyOptionsPerChannel(options);
        }
        int num_tracks = this.allevents.size();
        int[] instruments = new int[num_tracks];
        boolean[] keeptracks = new boolean[num_tracks];
        for (i = 0; i < num_tracks; ++i) {
            instruments[i] = 0;
            keeptracks[i] = true;
        }
        for (int tracknum2 = 0; tracknum2 < this.tracks.size(); ++tracknum2) {
            MidiTrack track = this.tracks.get(tracknum2);
            int realtrack = track.trackNumber();
            instruments[realtrack] = options.instruments[tracknum2];
            if (options.tracks[tracknum2] && !options.mute[tracknum2]) continue;
            keeptracks[realtrack] = false;
        }
        ArrayList<ArrayList<MidiEvent>> newevents = MidiFile.CloneMidiEvents(this.allevents);
        for (tracknum = 0; tracknum < newevents.size(); ++tracknum) {
            MidiEvent mevent = MidiFile.CreateTempoEvent(options.tempo);
            newevents.get(tracknum).add(0, mevent);
        }
        for (tracknum = 0; tracknum < newevents.size(); ++tracknum) {
            for (MidiEvent mevent : newevents.get(tracknum)) {
                int num = mevent.Notenumber + options.transpose;
                if (num < 0) {
                    num = 0;
                }
                if (num > 127) {
                    num = 127;
                }
                mevent.Notenumber = (byte)num;
                if (!options.useDefaultInstruments) {
                    mevent.Instrument = (byte)instruments[tracknum];
                }
                mevent.Tempo = options.tempo;
            }
        }
        if (options.pauseTime != 0) {
            newevents = MidiFile.StartAtPauseTime(newevents, options.pauseTime);
        }
        int count = 0;
        for (int tracknum3 = 0; tracknum3 < keeptracks.length; ++tracknum3) {
            if (!keeptracks[tracknum3]) continue;
            ++count;
        }
        ArrayList<ArrayList<MidiEvent>> result = new ArrayList<ArrayList<MidiEvent>>(count);
        i = 0;
        for (int tracknum4 = 0; tracknum4 < keeptracks.length; ++tracknum4) {
            if (!keeptracks[tracknum4]) continue;
            result.add(newevents.get(tracknum4));
            ++i;
        }
        return result;
    }

    public ArrayList<ArrayList<MidiEvent>> ApplyOptionsPerChannel(MidiOptions options) {
        int tracknum;
        int[] instruments = new int[16];
        boolean[] keepchannel = new boolean[16];
        for (int i = 0; i < 16; ++i) {
            instruments[i] = 0;
            keepchannel[i] = true;
        }
        for (int tracknum2 = 0; tracknum2 < this.tracks.size(); ++tracknum2) {
            MidiTrack track = this.tracks.get(tracknum2);
            int channel = track.getNotes().get(0).getChannel();
            instruments[channel] = options.instruments[tracknum2];
            if (options.tracks[tracknum2] && !options.mute[tracknum2]) continue;
            keepchannel[channel] = false;
        }
        ArrayList<ArrayList<MidiEvent>> newevents = MidiFile.CloneMidiEvents(this.allevents);
        for (tracknum = 0; tracknum < newevents.size(); ++tracknum) {
            MidiEvent mevent = MidiFile.CreateTempoEvent(options.tempo);
            newevents.get(tracknum).add(0, mevent);
        }
        for (tracknum = 0; tracknum < newevents.size(); ++tracknum) {
            for (MidiEvent mevent : newevents.get(tracknum)) {
                int num = mevent.Notenumber + options.transpose;
                if (num < 0) {
                    num = 0;
                }
                if (num > 127) {
                    num = 127;
                }
                mevent.Notenumber = (byte)num;
                if (!keepchannel[mevent.Channel]) {
                    mevent.Velocity = 0;
                }
                if (!options.useDefaultInstruments) {
                    mevent.Instrument = (byte)instruments[mevent.Channel];
                }
                mevent.Tempo = options.tempo;
            }
        }
        if (options.pauseTime != 0) {
            newevents = MidiFile.StartAtPauseTime(newevents, options.pauseTime);
        }
        return newevents;
    }

    public ArrayList<MidiTrack> ChangeMidiNotes(MidiOptions options) {
        ArrayList<MidiTrack> newtracks = new ArrayList<MidiTrack>();
        for (int track = 0; track < this.tracks.size(); ++track) {
            if (!options.tracks[track]) continue;
            newtracks.add(this.tracks.get(track).Clone());
        }
        TimeSignature time = this.timesig;
        if (options.time != null) {
            time = options.time;
        }
        MidiFile.RoundStartTimes(newtracks, options.combineInterval, this.timesig);
        MidiFile.RoundDurations(newtracks, time.getQuarter());
        if (options.twoStaffs) {
            newtracks = MidiFile.CombineToTwoTracks(newtracks, this.timesig.getMeasure());
        }
        if (options.shifttime != 0) {
            MidiFile.ShiftTime(newtracks, options.shifttime);
        }
        if (options.transpose != 0) {
            MidiFile.Transpose(newtracks, options.transpose);
        }
        return newtracks;
    }

    public static void ShiftTime(ArrayList<MidiTrack> tracks, int amount) {
        for (MidiTrack track : tracks) {
            for (MidiNote note : track.getNotes()) {
                note.setStartTime(note.getStartTime() + amount);
            }
        }
    }

    public static void Transpose(ArrayList<MidiTrack> tracks, int amount) {
        for (MidiTrack track : tracks) {
            for (MidiNote note : track.getNotes()) {
                note.setNumber(note.getNumber() + amount);
                if (note.getNumber() >= 0) continue;
                note.setNumber(0);
            }
        }
    }

    private static void FindHighLowNotes(ArrayList<MidiNote> notes, int measurelen, int startindex, int starttime, int endtime, PairInt pair) {
        int i = startindex;
        if (starttime + measurelen < endtime) {
            endtime = starttime + measurelen;
        }
        while (i < notes.size() && notes.get(i).getStartTime() < endtime) {
            if (notes.get(i).getEndTime() < starttime) {
                ++i;
                continue;
            }
            if (notes.get(i).getStartTime() + measurelen < starttime) {
                ++i;
                continue;
            }
            if (pair.high < notes.get(i).getNumber()) {
                pair.high = notes.get(i).getNumber();
            }
            if (pair.low > notes.get(i).getNumber()) {
                pair.low = notes.get(i).getNumber();
            }
            ++i;
        }
    }

    private static void FindExactHighLowNotes(ArrayList<MidiNote> notes, int startindex, int starttime, PairInt pair) {
        int i = startindex;
        while (notes.get(i).getStartTime() < starttime) {
            ++i;
        }
        while (i < notes.size() && notes.get(i).getStartTime() == starttime) {
            if (pair.high < notes.get(i).getNumber()) {
                pair.high = notes.get(i).getNumber();
            }
            if (pair.low > notes.get(i).getNumber()) {
                pair.low = notes.get(i).getNumber();
            }
            ++i;
        }
    }

    public static ArrayList<MidiTrack> SplitTrack(MidiTrack track, int measurelen) {
        ArrayList<MidiNote> notes = track.getNotes();
        int count = notes.size();
        MidiTrack top = new MidiTrack(1);
        MidiTrack bottom = new MidiTrack(2);
        ArrayList<MidiTrack> result = new ArrayList<MidiTrack>(2);
        result.add(top);
        result.add(bottom);
        if (count == 0) {
            return result;
        }
        int prevhigh = 76;
        int prevlow = 45;
        int startindex = 0;
        for (MidiNote note : notes) {
            int number;
            int lowExact;
            int highExact = lowExact = (number = note.getNumber());
            int low = lowExact;
            int high = lowExact;
            while (notes.get(startindex).getEndTime() < note.getStartTime()) {
                ++startindex;
            }
            PairInt pair = new PairInt();
            pair.high = high;
            pair.low = low;
            PairInt pairExact = new PairInt();
            pairExact.high = highExact;
            pairExact.low = lowExact;
            MidiFile.FindHighLowNotes(notes, measurelen, startindex, note.getStartTime(), note.getEndTime(), pair);
            MidiFile.FindExactHighLowNotes(notes, startindex, note.getStartTime(), pairExact);
            high = pair.high;
            low = pair.low;
            highExact = pairExact.high;
            lowExact = pairExact.low;
            if (highExact - number > 12 || number - lowExact > 12) {
                if (highExact - number <= number - lowExact) {
                    top.AddNote(note);
                } else {
                    bottom.AddNote(note);
                }
            } else if (high - number > 12 || number - low > 12) {
                if (high - number <= number - low) {
                    top.AddNote(note);
                } else {
                    bottom.AddNote(note);
                }
            } else if (highExact - lowExact > 12) {
                if (highExact - number <= number - lowExact) {
                    top.AddNote(note);
                } else {
                    bottom.AddNote(note);
                }
            } else if (high - low > 12) {
                if (high - number <= number - low) {
                    top.AddNote(note);
                } else {
                    bottom.AddNote(note);
                }
            } else if (prevhigh - number <= number - prevlow) {
                top.AddNote(note);
            } else {
                bottom.AddNote(note);
            }
            if (high - low <= 12) continue;
            prevhigh = high;
            prevlow = low;
        }
        Collections.sort(top.getNotes(), track.getNotes().get(0));
        Collections.sort(bottom.getNotes(), track.getNotes().get(0));
        return result;
    }

    public static MidiTrack CombineToSingleTrack(ArrayList<MidiTrack> tracks) {
        MidiTrack result = new MidiTrack(1);
        if (tracks.size() == 0) {
            return result;
        }
        if (tracks.size() == 1) {
            MidiTrack track = tracks.get(0);
            for (MidiNote note : track.getNotes()) {
                result.AddNote(note);
            }
            return result;
        }
        int[] noteindex = new int[tracks.size() + 1];
        int[] notecount = new int[tracks.size() + 1];
        for (int tracknum = 0; tracknum < tracks.size(); ++tracknum) {
            noteindex[tracknum] = 0;
            notecount[tracknum] = tracks.get(tracknum).getNotes().size();
        }
        MidiNote prevnote = null;
        while (true) {
            MidiNote lowestnote = null;
            int lowestTrack = -1;
            for (int tracknum = 0; tracknum < tracks.size(); ++tracknum) {
                MidiTrack track = tracks.get(tracknum);
                if (noteindex[tracknum] >= notecount[tracknum]) continue;
                MidiNote note = track.getNotes().get(noteindex[tracknum]);
                if (lowestnote == null) {
                    lowestnote = note;
                    lowestTrack = tracknum;
                    continue;
                }
                if (note.getStartTime() < lowestnote.getStartTime()) {
                    lowestnote = note;
                    lowestTrack = tracknum;
                    continue;
                }
                if (note.getStartTime() != lowestnote.getStartTime() || note.getNumber() >= lowestnote.getNumber()) continue;
                lowestnote = note;
                lowestTrack = tracknum;
            }
            if (lowestnote == null) break;
            int n = lowestTrack;
            noteindex[n] = noteindex[n] + 1;
            if (prevnote != null && prevnote.getStartTime() == lowestnote.getStartTime() && prevnote.getNumber() == lowestnote.getNumber()) {
                if (lowestnote.getDuration() <= prevnote.getDuration()) continue;
                prevnote.setDuration(lowestnote.getDuration());
                continue;
            }
            result.AddNote(lowestnote);
            prevnote = lowestnote;
        }
        return result;
    }

    public static ArrayList<MidiTrack> CombineToTwoTracks(ArrayList<MidiTrack> tracks, int measurelen) {
        MidiTrack single = MidiFile.CombineToSingleTrack(tracks);
        ArrayList<MidiTrack> result = MidiFile.SplitTrack(single, measurelen);
        ArrayList<MidiEvent> lyrics = new ArrayList<MidiEvent>();
        for (MidiTrack track : tracks) {
            if (track.getLyrics() == null) continue;
            lyrics.addAll(track.getLyrics());
        }
        if (lyrics.size() > 0) {
            Collections.sort(lyrics, (Comparator)lyrics.get(0));
            result.get(0).setLyrics(lyrics);
        }
        return result;
    }

    private static void CheckStartTimes(ArrayList<MidiTrack> tracks) {
        for (MidiTrack track : tracks) {
            int prevtime = -1;
            for (MidiNote note : track.getNotes()) {
                if (note.getStartTime() < prevtime) {
                    throw new MidiFileException("Internal parsing error", 0);
                }
                prevtime = note.getStartTime();
            }
        }
    }

    public static void RoundStartTimes(ArrayList<MidiTrack> tracks, int millisec, TimeSignature time) {
        ListInt starttimes = new ListInt();
        for (MidiTrack track : tracks) {
            for (MidiNote note : track.getNotes()) {
                starttimes.add(note.getStartTime());
            }
        }
        starttimes.sort();
        int interval = time.getQuarter() * millisec * 1000 / time.getTempo();
        for (int i = 0; i < starttimes.size() - 1; ++i) {
            if (starttimes.get(i + 1) - starttimes.get(i) > interval) continue;
            starttimes.set(i + 1, starttimes.get(i));
        }
        MidiFile.CheckStartTimes(tracks);
        for (MidiTrack track : tracks) {
            int i = 0;
            for (MidiNote note : track.getNotes()) {
                while (i < starttimes.size() && note.getStartTime() - interval > starttimes.get(i)) {
                    ++i;
                }
                if (note.getStartTime() <= starttimes.get(i) || note.getStartTime() - starttimes.get(i) > interval) continue;
                note.setStartTime(starttimes.get(i));
            }
            Collections.sort(track.getNotes(), track.getNotes().get(0));
        }
    }

    public static void RoundDurations(ArrayList<MidiTrack> tracks, int quarternote) {
        for (MidiTrack track : tracks) {
            MidiNote prevNote = null;
            for (int i = 0; i < track.getNotes().size() - 1; ++i) {
                MidiNote note1 = track.getNotes().get(i);
                if (prevNote == null) {
                    prevNote = note1;
                }
                MidiNote note2 = note1;
                for (int j = i + 1; j < track.getNotes().size(); ++j) {
                    note2 = track.getNotes().get(j);
                    if (note1.getStartTime() < note2.getStartTime()) break;
                }
                int maxduration = note2.getStartTime() - note1.getStartTime();
                int dur = 0;
                if (quarternote <= maxduration) {
                    dur = quarternote;
                } else if (quarternote / 2 <= maxduration) {
                    dur = quarternote / 2;
                } else if (quarternote / 3 <= maxduration) {
                    dur = quarternote / 3;
                } else if (quarternote / 4 <= maxduration) {
                    dur = quarternote / 4;
                }
                if (dur < note1.getDuration()) {
                    dur = note1.getDuration();
                }
                if (prevNote.getStartTime() + prevNote.getDuration() == note1.getStartTime() && prevNote.getDuration() == note1.getDuration()) {
                    dur = note1.getDuration();
                }
                note1.setDuration(dur);
                if (track.getNotes().get(i + 1).getStartTime() == note1.getStartTime()) continue;
                prevNote = note1;
            }
        }
    }

    private static ArrayList<MidiTrack> SplitChannels(MidiTrack origtrack, ArrayList<MidiEvent> events) {
        int[] channelInstruments = new int[16];
        for (MidiEvent midiEvent : events) {
            if (midiEvent.EventFlag != -64) continue;
            channelInstruments[midiEvent.Channel] = midiEvent.Instrument;
        }
        channelInstruments[9] = 128;
        ArrayList<MidiTrack> result = new ArrayList<MidiTrack>();
        for (MidiNote note : origtrack.getNotes()) {
            boolean foundchannel = false;
            for (MidiTrack track : result) {
                if (note.getChannel() != track.getNotes().get(0).getChannel()) continue;
                foundchannel = true;
                track.AddNote(note);
            }
            if (foundchannel) continue;
            MidiTrack track = new MidiTrack(result.size() + 1);
            track.AddNote(note);
            track.setInstrument(channelInstruments[note.getChannel()]);
            result.add(track);
        }
        ArrayList<MidiEvent> arrayList = origtrack.getLyrics();
        if (arrayList != null) {
            for (MidiEvent lyricEvent : arrayList) {
                for (MidiTrack track : result) {
                    if (lyricEvent.Channel != track.getNotes().get(0).getChannel()) continue;
                    track.AddLyric(lyricEvent);
                }
            }
        }
        return result;
    }

    public ListInt GuessMeasureLength() {
        ListInt result = new ListInt();
        int pulses_per_second = (int)(1000000.0 / (double)this.timesig.getTempo() * (double)this.timesig.getQuarter());
        int minmeasure = pulses_per_second / 2;
        int maxmeasure = pulses_per_second * 4;
        int firstnote = this.timesig.getMeasure() * 5;
        for (MidiTrack track : this.tracks) {
            if (firstnote <= track.getNotes().get(0).getStartTime()) continue;
            firstnote = track.getNotes().get(0).getStartTime();
        }
        int interval = this.timesig.getQuarter() * 60000 / this.timesig.getTempo();
        block1: for (MidiTrack track : this.tracks) {
            int prevtime = 0;
            for (MidiNote note : track.getNotes()) {
                if (note.getStartTime() - prevtime <= interval) continue;
                prevtime = note.getStartTime();
                int time_from_firstnote = note.getStartTime() - firstnote;
                if ((time_from_firstnote = time_from_firstnote / 4 * 4) < minmeasure) continue;
                if (time_from_firstnote > maxmeasure) continue block1;
                if (result.contains(time_from_firstnote)) continue;
                result.add(time_from_firstnote);
            }
        }
        result.sort();
        return result;
    }

    public int EndTime() {
        int lastStart = 0;
        for (MidiTrack track : this.tracks) {
            if (track.getNotes().size() == 0) continue;
            int last = track.getNotes().get(track.getNotes().size() - 1).getStartTime();
            lastStart = Math.max(last, lastStart);
        }
        return lastStart;
    }

    public boolean hasLyrics() {
        for (MidiTrack track : this.tracks) {
            if (track.getLyrics() == null) continue;
            return true;
        }
        return false;
    }

    public static boolean hasMidiHeader(byte[] data) {
        try {
            String s = new String(data, 0, 4, "US-ASCII");
            return s.equals("MThd");
        }
        catch (UnsupportedEncodingException e) {
            return false;
        }
    }

    public String toString() {
        String result = "Midi File tracks=" + this.tracks.size() + " quarter=" + this.quarternote + "\n";
        result = result + this.timesig.toString() + "\n";
        for (MidiTrack track : this.tracks) {
            result = result + track.toString();
        }
        return result;
    }

    public static void main2(String[] args) {
    }
}

